global TIKI_BONE_OFFSET_MANTISSA_BITS = 9
global TIKI_BONE_OFFSET_MAX_SIGNED_VALUE = (bit.shift 1 TIKI_BONE_OFFSET_MANTISSA_BITS) - 1
global TIKI_BONE_OFFSET_SIGNED_SHIFT = (15 - TIKI_BONE_OFFSET_MANTISSA_BITS)
global TIKI_BONE_OFFSET_MULTIPLIER = ((bit.shift 1 TIKI_BONE_OFFSET_SIGNED_SHIFT) - 1)
global TIKI_BONE_OFFSET_MULTIPLIER_RECIPROCAL = 1.0 / TIKI_BONE_OFFSET_MULTIPLIER

global TIKI_BONE_QUAT_FRACTIONAL_BITS = 15
global TIKI_BONE_QUAT_MULTIPLIER = (bit.shift 1 TIKI_BONE_QUAT_FRACTIONAL_BITS) - 1
global TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL = 1.0 / TIKI_BONE_QUAT_MULTIPLIER

struct SKB_Header
(
	ident,
	version,
	name,
	numSurfaces,
	numBones,
	ofsBones,
	ofsSurfaces,
	ofsBaseFrame,
	ofsEnd,
	
	fn LoadFromStream bstream =
	(
		ident 			 = ReadLong bstream
		version 		 = ReadLong bstream
		name 			 = ReadFixedString bstream 64
		numSurfaces 	 = ReadLong bstream
		numBones 		 = ReadLong bstream
		ofsBones 		 = ReadLong bstream
		ofsSurfaces 	 = ReadLong bstream
		if version == 4 then
			ofsBaseFrame = ReadLong bstream
		ofsEnd 			 = ReadLong bstream
	),
	
	fn IsValid =
	(
		local identStr = LongToString ident
		if( LongToString ident ) != "SKL " then
			return false
		-- version = 4 for ef2 models
		-- version = 3 for fakk2 models
		if version != 4 then
		(
			if version != 3 then
			(
				ShowError "Incorrect version! Found " + version + " but should be 3 of 4"
				return false
			)
		)
		true
	),
	
	fn DebugOut =
	(
		format "SKB Header\n---------\n"
		format "Ident: %, version: %, name: %\n" ident version name
		format "numSurfaces: %, numBones: %\n" numSurfaces numBones
		format "ofsBones: %, ofsSurfaces: % ," ofsBones ofsSurfaces
		if version == 4 then
			format "ofsBaseFrame: % ," ofsBaseFrame
		format "ofsEnd: %\n\n" ofsEnd
		ok
	)
)

struct SKB_BoneName
(
	parent,
	flags,
	name,
	
	fn LoadFromStream bstream =
	(
		parent 	= ReadLong bstream
		flags 	= ReadLong bstream
		name 	= ReadFixedString bstream 64
		
		if parent != -1 then
			parent += 1
	),
	
	fn DebugOut =
	(
		format "SKB_BoneName\n--------\n"
		format "parent: %\n" parent
		format "flags: %\n" flags
		format "name: %\n\n" name
	)
)

struct SKB_Triangle
(
	indices,
	
	fn LoadFromStream bstream =
	(
		local v1 = ReadLong bstream #unsigned
		local v2 = ReadLong bstream #unsigned
		local v3 = ReadLong bstream #unsigned
		indices = [v3 + 1, v2 + 1, v1 + 1]
	)
)

struct SKB_Weight
(
	boneIndex,
	boneWeight,
	offset,
	
	fn LoadFromStream bstream =
	(
		boneIndex = ReadLong bstream
		boneWeight = ReadFloat bstream
		local x = ReadFloat bstream
		local y = ReadFloat bstream
		local z = ReadFloat bstream
		offset = [x, y, z]
	)
)

struct SKB_Vertex
(
	normal,
	st,
	numWeights,
	weights,
	
	fn LoadFromStream bstream =
	(
		local x = ReadFloat bstream
		local y = ReadFloat bstream
		local z = ReadFloat bstream
		normal = [x, y, z]
		local s = ReadFloat bstream
		local t = ReadFloat bstream
		st = [s, 1.0 - t, 0.0]
		
		numWeights = ReadLong bstream
		
		weights = #()
		for i = 1 to numWeights do
		(
			local curWeight = SKB_Weight()
			curWeight.LoadFromStream bstream
			append weights (copy curWeight)
		)
	)
)

struct SKB_Surface
(
	ident,
	name,
	numTris,
	numVerts,
	minLod,
	ofsTris,
	ofsVerts,
	ofsCollapse,
	ofsEnd,
	
	-- Internal
	ofsStart,
	tris,
	verts,
	collapseMap,
	surfMesh,
	skinMod,
	skinModBoneMap,
	
	
	fn LoadFromStream bstream =
	(
		ofsStart = ftell bstream
	
		ident			= ReadLong bstream
		name			= ReadFixedString bstream 64
		numTris			= ReadLong bstream
		numVerts		= ReadLong bstream
		minLod			= ReadLong bstream
		ofsTris			= ReadLong bstream
		ofsVerts		= ReadLong bstream
		ofsCollapse		= ReadLong bstream
		ofsEnd			= ReadLong bstream
	),
	
	fn IsValid =
	(
		if (LongToString ident) != "SKL " then
		(
			return false
		)
		true
	),
	
	fn DebugOut =
	(
		format "SKB_Surface\n-----------\n"
		format "ident: %, name: %\n" ident name
		format "numTris: %, numVerts: %, minLod: %\n\n" numTris numVerts minLod
	),
	
	fn LoadData bstream =
	(
		fseek bstream (ofsStart + ofsTris) #seek_set
		
		tris = #()
		local curTris = SKB_Triangle()
		for i = 1 to numTris do
		(
			curTris.LoadFromStream bstream
			append tris curTris.indices
		)
		
		fseek bstream (ofsStart + ofsVerts) #seek_set
		
		verts = #()
		local curVert = SKB_Vertex()
		for i = 1 to numVerts do
		(
			curVert.LoadFromStream bstream
			append verts (copy curVert)
		)
		
		fseek bstream (ofsStart + ofsCollapse) #seek_set
		
		collapseMap = #()
		for i = 1 to numVerts do
		(
			append collapseMap (ReadLong bstream)
		)
	),
	
	fn CreateMesh matID=
	(
		surfMesh = mesh name:name numVerts:numVerts faces:numTris
		
		-- set faces
		setMesh surfMesh faces:tris
		for t = 1 to numTris do
		(
			setFaceMatID surfMesh t matID
		)
		
		-- set texCoords
		local tverts = #()
		for i = 1 to numVerts do
		(
			append tverts verts[ i ].st
		)
		setMesh surfMesh tverts:tverts
		
		-- set texture faces
		buildTVFaces surfMesh false
		for i = 1 to numTris do
		(
			setTVFace surfMesh i  tris[ i ]
		)
		
		surfMesh
	)
)

struct SKB_BaseFrame
(
	myQuat,
	offset,
	
	-- Internal
	matrix,
	
	-- for test purposes
	shortQuat,
	shortOffset,
	
	
	fn LoadFromStream bstream =
	(
		local x = ReadShort bstream
		local y = ReadShort bstream
		local z = ReadShort bstream
		local w = ReadShort bstream
		shortQuat = quat x y z w
		
		x = (x as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		y = (y as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		z = (z as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		w = (w as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		myQuat = quat x y z w
		
		x = ReadShort bstream
		y = ReadShort bstream
		z = ReadShort bstream
		shortOffset = point3 x y z
		
		x = (x as float) * TIKI_BONE_OFFSET_MULTIPLIER_RECIPROCAL
		y = (y as float) * TIKI_BONE_OFFSET_MULTIPLIER_RECIPROCAL
		z = (z as float) * TIKI_BONE_OFFSET_MULTIPLIER_RECIPROCAL
		offset = point3 x y z
		
		local notUsed = ReadShort bstream
	),
	
	fn DebugOut =
	(
		format "shortQuat: %\n" shortQuat
		format "shortOffset: %\n\n" shortOffset
		format "myQuat: %\n" myQuat
		format "offset:%\n\n" offset
	)
)	

struct SKA_Header
(
	ident,
	version,
	name,
	type,
	numFrames,
	numBones,
	totalTime,
	frameTime,
	totalDelta,
	ofsBones,
	ofsFrames,

	fn LoadFromStream bstream =
	(
		ident			= ReadLong bstream
		version			= ReadLong bstream
		name 			= ReadFixedString bstream 64
		type			= ReadLong bstream
		numFrames		= ReadLong bstream
		numBones		= ReadLong bstream
		totalTime		= ReadFloat bstream
		frameTime		= ReadFloat bstream
		local x			= ReadFloat bstream
		local y			= ReadFloat bstream
		local z			= ReadFloat bstream
		totalDelta = [x, y, z]
		if version == 4 then
		(
			ofsBones	= ReadLong bstream
		)
		ofsFrames		= ReadLong bstream
	),

	fn IsValid =
	(
		if (LongToString ident) != "SKAN" then
		( 	
			return false
		)
		if version != 4 then
		(
			if version != 3 then
			(
				ShowError "Incorrect version! Found " + (version as string) + " but should be 3 of 4"
				return false
			)
		)
		if numFrames < 1 then
		(
			ShowError "SKA does not have any frames!"
			return false
		)
		true
	),
	
	fn DebugOut = 
	(
		format "SKA_Header\n---------\n"
		format "ident: %, version: %, name: %\n" ident version name
		format "type: %, numFrames: %, numBones: %\n" type numFrames numBones
		format "totalTime: %, frameTime: %, totalDelta: %\n" totalTime frameTime totalDelta
		format "ofsFrames: %\n\n" ofsFrames
		ok
	)
)

struct SKA_Bone
(
	myQuat,
	offset,
	
	-- Internal
	matrix,
	
	-- for test purposes
	shortQuat,
	shortOffset,
	
	
	fn LoadFromStream bstream =
	(
		local x = ReadShort bstream
		local y = ReadShort bstream
		local z = ReadShort bstream
		local w = ReadShort bstream
		shortQuat = quat x y z w
		
		x = (x as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		y = (y as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		z = (z as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		w = (w as float) * TIKI_BONE_QUAT_MULTIPLIER_RECIPROCAL
		myQuat = quat x y z w
		
		x = ReadShort bstream
		y = ReadShort bstream
		z = ReadShort bstream
		shortOffset = point3 x y z
		
		x = (x as float) * TIKI_BONE_OFFSET_MULTIPLIER_RECIPROCAL
		y = (y as float) * TIKI_BONE_OFFSET_MULTIPLIER_RECIPROCAL
		z = (z as float) * TIKI_BONE_OFFSET_MULTIPLIER_RECIPROCAL
		offset = point3 x y z
		
		local notUsed = ReadShort bstream
	),
	
	fn DebugOut =
	(
		format "%\n" myQuat
		format "%\n\n" offset
	)
)

struct SKA_Frame
(
	bBoxMin,
	bBoxMax,
	radius,
	delta,
	
	bones = #(),
	
	fn LoadFromStream bstream =
	(
		local x = ReadFloat bstream
		local y = ReadFloat bstream
		local z = ReadFloat bstream
		bBoxMin = [x, y, z]
		x = ReadFloat bstream
		y = ReadFloat bstream
		z = ReadFloat bstream
		bBoxMax = [x, y, z]
		radius = ReadFloat bstream
		x = ReadFloat bstream
		y = ReadFloat bstream
		z = ReadFloat bstream
		delta = [x, y, z]
	),
	
	fn DebugOut =
	(
		format "SKA_Frame:\n"
		format "bBoxMin: %\n" bBoxMin
		format "bBoxMax: %\n" bBoxMax
		format "radius: %\n" radius
		format "delta: %\n\n" delta
	)
)
		
		
		
		
		
		